/**
 * @file    AndroidDependencyInfo.cs
 * 
 * @author  jschoi
 * @date    2024~
 * @copyright Copyright © Com2uS Platform Corporation. All Right Reserved.
 * @defgroup Hive.Unity.Editor
 * @{
 * @brief HIVE External Dependency Info <br/><br/>
 */

using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEditor;
using System;
using System.IO;
using System.ComponentModel;
using System.Reflection;
using System.Xml;

namespace Hive.Unity.Editor
{    
    using ExternalDependencyDictionary = Dictionary<ExternalDependencyType, bool>;

    public class AndroidDependencyInfo {

        // Android 플랫폼 버전 파일 구조화
        [System.Serializable]
        public class VersionFileInfo
        {
            public string type;
            public string os;
            public string version;
        }

        // Android 플랫폼 버전 정보 획득
        // - /Assets/Hive_SDK_v4/Version 에 있는 HiveSDKPlatformAndroidVersion.json 파일에서 "version" 키의 값을 반환합니다.
        public static string getAndroidDependencyVersion()
        {
            if(File.Exists(ExternalDependencyConst.hiveAndroidVersionJsonPath)) {
                try
                {
                    var json = System.IO.File.ReadAllText(ExternalDependencyConst.hiveAndroidVersionJsonPath);
                    var versionFileInfo = JsonUtility.FromJson<VersionFileInfo>(json);
                    return versionFileInfo.version;
                }
                catch (Exception e)
                {
                    return "";
                }
            }
            return "";
        }

        // <androidPackages> 태그 이하 모든 정보를 수집해서 반환
        private static HashSet<string> getAndroidPackagesSet(ExternalDependencyDictionary dictionary, AndroidDependencyDataType dataType) {
            HashSet<string> set = new HashSet<string>();    // HashSet 중복 허용하지 않음

            // dictionary에서 true 설정된 provider 정보 획득
            foreach(KeyValuePair<ExternalDependencyType, bool> pair in dictionary) {
                if(pair.Value == true)  {

                    AndroidDependencyXmlInfo dependencyXmlInfo = getDependencyInfo(pair.Key);
                    if(dependencyXmlInfo != null)  {
                        
                        switch(dataType) {
                            case AndroidDependencyDataType.AndroidRepository:
                                foreach(string data in dependencyXmlInfo.androidRepository) {
                                    set.Add(data);
                                }
                                break;
                            case AndroidDependencyDataType.AndroidPackageSpec:
                                foreach(string data in dependencyXmlInfo.androidPackageSpec) {
                                    set.Add(data);
                                }
                                break;
                        }
                    }
                }
            }

            // HiveSDK default 정보 획득(ExternalDependecyType에 속하지 않아 항상 포함되도록)
            HiveDefault hiveDefault = new HiveDefault(ExternalDependencyUtil.isSampleBuild());

            switch(dataType) {
                case AndroidDependencyDataType.AndroidRepository:
                    foreach(string data in hiveDefault.androidRepository) {
                        set.Add(data);
                    }
                    break;
                case AndroidDependencyDataType.AndroidPackageSpec:
                    foreach(string data in hiveDefault.androidPackageSpec) {
                        set.Add(data);
                    }
                    break;
            }

            return set;
        }

        public static string getAndroidPackagesData(ExternalDependencyDictionary dictionary) {
            HashSet<string> androidRepository = new HashSet<string>();
            HashSet<string> androidPackageSpec = new HashSet<string>();

            androidRepository = getAndroidPackagesSet(dictionary, AndroidDependencyDataType.AndroidRepository);
            androidPackageSpec = getAndroidPackagesSet(dictionary, AndroidDependencyDataType.AndroidPackageSpec);

            string xmlText = makeUpXmlString(androidRepository, androidPackageSpec);

            return xmlText;
        }

        private static string makeUpXmlString(HashSet<string> adRepository, HashSet<string> adPackageSpec) {
            string space = "    ";
            string xmlText = "";

            // repositories
            xmlText += space + "<androidPackages>\n";
            xmlText += space + space + "<repositories>\n";

            foreach(string data in adRepository) {
                if(data != null && !data.Equals("")) {
                    xmlText += space + space + space;
                    xmlText += "<repository>" + data + "</repository>\n";
                }
            }
            xmlText += space + space + "</repositories>\n";

            // androidPackages
            foreach(string data in adPackageSpec) {
                if(ExternalDependencyUtil.isSampleBuild()) {
                    
                    if(data != null && !data.Equals("")) {
                        // 샘플 기준 빌드 작성 (Android의 extra-google-m2repository 설정 추가 여부 확인)
                        string tmpText = addAndroidSdkPackageIdInfoForSample(data);
                        if(!tmpText.Equals("")) {
                            xmlText += tmpText;
                        }
                        else {
                            xmlText += space + space;
                            xmlText += "<androidPackage spec=\"" + data + "\"/>\n";
                        }
                    }
                } else {
                    // 배포 기준 빌드 작성
                    if(data != null && !data.Equals("")) {
                        xmlText += space + space;
                        xmlText += "<androidPackage spec=\"" + data + "\"/>\n";
                    }
                }
            }
            xmlText += space + "</androidPackages>\n";

            return xmlText;
        }

        private static string addAndroidSdkPackageIdInfoForSample(string spec) {
            string space = "    ";
            string xmlText = "";
            List<string> targetSpec = new List<string>() {"com.google.android.play:review",
                                                            "com.google.android.play:app-update",
                                                            "com.google.android.play:age-signals",
                                                            "com.google.firebase:firebase-analytic",
                                                            "com.google.android.gms:play-services-ads-identifier",
                                                            "com.google.android.gms:play-services-analytics:",
                                                            "com.google.firebase:firebase-messaging",
                                                            "com.google.android.gms:play-services-games-v2",
                                                            "com.google.android.recaptcha:recaptcha"};
            
            foreach(string targetData in targetSpec) {
                if(spec.Contains(targetData)) {
                    xmlText += space + space + "<androidPackage spec=\"" + spec + "\">\n";
                    xmlText += space + space + space + "<androidSdkPackageIds>\n";
                    xmlText += space + space + space + space + "<androidSdkPackageId>extra-google-m2repository</androidSdkPackageId>\n";
                    xmlText += space + space + space + "</androidSdkPackageIds>\n";
                    xmlText += space + space + "</androidPackage>\n";
                }
            }

            return xmlText;
        }

        // DependencyInfo를 상속받은 클래스 중 type에 해당하는 클래스 반환
        private static AndroidDependencyXmlInfo getDependencyInfo(ExternalDependencyType type) {
            switch(type) {
                case ExternalDependencyType.Hercules:
                    return new Hercules(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.AnalyticsConsentMode:
                    return new AnalyticsConsentMode(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MatchMaking:
                    return new HiveMatchMaking(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Chat:
                    return new HiveChat(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.DataStore:
                    return new HiveDataStore(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveDeviceManagement:
                    return new HiveDeviceManagement(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.GoogleInAppReview:
                    return new HiveGoogleInAppReview(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.GoogleInAppUpdate:
                    return new HiveGoogleInAppUpdate(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HivePromotion:
                    return new HivePromotion(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveRealNameVerification:
                    return new HiveRealNameVerification(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketAmazon:
                    return new MarketAmazon(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveAgeRangeAmazon:
                    return new HiveAgeRangeAmazon(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketApple:
                    return new MarketApple(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveAgeRangeApple:
                    return new HiveAgeRangeApple(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketGoogle:
                    return new MarketGoogle(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveAgeRangeGoogle:
                    return new HiveAgeRangeGoogle(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketHiveStore:
                    return new MarketHiveStore(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketHuawei:
                    return new MarketHuawei(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketLebi:
                    return new MarketLebi(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketOneStore:
                    return new MarketOneStore(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketSamsung:
                    return new MarketSamsung(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.MarketXiaomi:
                    return new MarketXiaomi(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Adjust:
                    return new ProviderAdjust(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Airbridge:
                    return new ProviderAirbridge(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Apple:
                    return new ProviderApple(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.AppleSignIn:
                    return new ProviderAppleSignIn(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.AppsFlyer:
                    return new ProviderAppsFlyer(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Facebook:
                    return new ProviderFacebook(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Firebase:
                    return new ProviderFirebase(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Google:
                    return new ProviderGoogle(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.GooglePlayGames:
                    return new ProviderGooglePlayGames(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveMembership:
                    return new ProviderHiveMembership(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Huawei:
                    return new ProviderHuawei(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Line:
                    return new ProviderLine(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.QQ:
                    return new ProviderQQ(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Recaptcha:
                    return new ProviderRecaptcha(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Singular:
                    return new ProviderSingular(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.VK:
                    return new ProviderVK(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Wechat:
                    return new ProviderWechat(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Weverse:
                    return new ProviderWeverse(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.X:
                    return new ProviderX(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Steam:
                    return new ProviderSteam(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Telegram:
                    return new ProviderTelegram(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.Xiaomi:
                    return new ProviderXiaomi(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.PushAmazonAdm:
                    return new PushAmazonAdm(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.PushGoogleFcm:
                    return new PushGoogleFcm(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveAdultConfirm:
                    return new HiveAdultConfirm(ExternalDependencyUtil.isSampleBuild());
                case ExternalDependencyType.HiveIdentityVerification:
                    return new HiveIdentityVerification(ExternalDependencyUtil.isSampleBuild());
                default:
                    return null;
            }
        }

        public abstract class AndroidDependencyXmlInfo {
            public List<string> androidRepository;
            public List<string> androidPackageSpec;
        }

        public enum AndroidDependencyDataType {
            AndroidRepository,
            AndroidPackageSpec
        }

        public class AndroidExternalLibraryVersion {
            public class HiveModule {
                public static readonly string playReview = "2.0.2";
                public static readonly string playAppUpdate = "2.1.0";
                public static readonly string recaptcha = "18.7.1";
                public static readonly string ageSignals = "0.0.2";
            }
            public class Auth {
                public static readonly string wechatSdkMta = "6.8.0";
                public static readonly string facebook = "18.0.1";
                public static readonly string credentials = "1.3.0";
                public static readonly string credentialPlayServicesAuth = "1.3.0";
                public static readonly string identityGoogleId = "1.1.1";
                public static readonly string playServicesAuth = "21.3.0";
                public static readonly string playServicesAdsIdentifier = "18.2.0";
                public static readonly string playServicesAnalytics = "18.1.1";
                public static readonly string firebaseMessaging = "24.1.1";
                public static readonly string playServicesGamesV2 = "21.0.0";
                public static readonly string hmsHwid = "6.12.0.300";
                public static readonly string line = "5.11.1";
                public static readonly string qqOkhttp = "4.12.0";
                public static readonly string gson = "2.10.1";      // SDK v1 샘플앱 디펜던시와 동기화 (2.13.1 사용 시 Unity 2022.x에서 빌드 불가 이슈 발생 - exclude com.google.errorprone 제거 필요)
                public static readonly string googleZxing = "3.2.0";
                public static readonly string xiaomiOauth = "3.4.6";
                public static readonly string xiaomiOnetrack = "2.2.5";
            }
            public class Push {
                public static readonly string firebaseMessaging = Auth.firebaseMessaging;
            }
            public class Market {
                public static readonly string amazonAppStore = "3.0.8";
                public static readonly string billingClient = "8.0.0";
                public static readonly string hmsIap = "6.13.0.300";
                public static readonly string alipaySdk = "15.8.17";
                public static readonly string wechatSdkMta = Auth.wechatSdkMta;
                public static readonly string onestorecorpSdkIap = "21.02.01";
            }
            public class Analytics {
                public static readonly string adjust = "5.4.2";
                public static readonly string airbridge = "4.7.1";
                public static readonly string appsflyerSdk = "6.17.3";
                public static readonly string firebase = "23.0.0";
                public static readonly string singular = "12.10.0";
            }
            public class Etc {
                public static readonly string youTubeHelper = "1.0.1";
                public static readonly string installReferrer = "2.2";
                public static readonly string androidxWorkRuntimeKtx = "2.10.3";
            }
        }

        // HiveDefault : ExternalDependencyType에 없지만 기본 포함 필요
        // (구 HIVEDefsult, 구 HIVESDKDependencies)
        private class HiveDefault : AndroidDependencyXmlInfo {
            public HiveDefault() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {$"com.com2us.android.hive:youtube-helper:{AndroidExternalLibraryVersion.Etc.youTubeHelper}",
                                                        $"com.android.installreferrer:installreferrer:{AndroidExternalLibraryVersion.Etc.installReferrer}",
                                                        $"com.com2us.android.hive:hive-sdk:{getAndroidDependencyVersion()}",
                                                        $"com.com2us.android.hive:hive-plugin-java:{getAndroidDependencyVersion()}"};
            }
            public HiveDefault(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://central.sonatype.com/repository/maven-snapshots/"};
                    androidPackageSpec = new List<string>() {$"com.com2us.android.hive:youtube-helper:{AndroidExternalLibraryVersion.Etc.youTubeHelper}",
                                                            $"com.android.installreferrer:installreferrer:{AndroidExternalLibraryVersion.Etc.installReferrer}",
                                                            $"androidx.work:work-runtime-ktx:{AndroidExternalLibraryVersion.Etc.androidxWorkRuntimeKtx}"};
                }
            }
        }

        class Hercules : AndroidDependencyXmlInfo {
            public Hercules() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-hercules:" + getAndroidDependencyVersion()};
            }
            public Hercules(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {"com.com2us.android.gamesec:hercules:20250527"};
                }
            }
        }
        
        class AnalyticsConsentMode : AndroidDependencyXmlInfo {
            public AnalyticsConsentMode() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-analytics-consent-mode:" + getAndroidDependencyVersion()};
            }
            public AnalyticsConsentMode(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class HiveDataStore : AndroidDependencyXmlInfo {
            public HiveDataStore() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-datastore:" + getAndroidDependencyVersion()};
            }
            public HiveDataStore(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class HiveDeviceManagement : AndroidDependencyXmlInfo {
            public HiveDeviceManagement() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-device-management:" + getAndroidDependencyVersion()};
            }
            public HiveDeviceManagement(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class HiveAdultConfirm : AndroidDependencyXmlInfo
        {
            public HiveAdultConfirm()
            {
                androidRepository = new List<string>() { "https://repo.maven.apache.org/maven2" };
                androidPackageSpec = new List<string>() { "com.com2us.android.hive:hive-authv4-adult-confirm:" + getAndroidDependencyVersion() };
            }
            public HiveAdultConfirm(bool sampleBuild) : this()
            {
                if (sampleBuild)
                {
                    androidRepository = new List<string>() { "" };
                    androidPackageSpec = new List<string>() { "" };
                }
            }
        }
        class HiveIdentityVerification : AndroidDependencyXmlInfo {
            public HiveIdentityVerification() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-identity-verification:" + getAndroidDependencyVersion()};
            }
            public HiveIdentityVerification(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class HiveGoogleInAppReview : AndroidDependencyXmlInfo
        {
            public HiveGoogleInAppReview()
            {
                androidRepository = new List<string>() { "https://repo.maven.apache.org/maven2" };
                androidPackageSpec = new List<string>() { "com.com2us.android.hive:hive-promotion-google-inappreview:" + getAndroidDependencyVersion() };
            }
            public HiveGoogleInAppReview(bool sampleBuild) : this()
            {
                if (sampleBuild)
                {
                    androidRepository = new List<string>() { "https://repo.maven.apache.org/maven2" };
                    androidPackageSpec = new List<string>() { $"com.google.android.play:review:{AndroidExternalLibraryVersion.HiveModule.playReview}" };
                }
            }
        }

        class HiveGoogleInAppUpdate : AndroidDependencyXmlInfo {
            public HiveGoogleInAppUpdate() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-google-inappupdate:" + getAndroidDependencyVersion()};
            }
            public HiveGoogleInAppUpdate(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                    androidPackageSpec = new List<string>() {$"com.google.android.play:app-update:{AndroidExternalLibraryVersion.HiveModule.playAppUpdate}"};
                }
            }
        }

        class HiveMatchMaking : AndroidDependencyXmlInfo {
            public HiveMatchMaking() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-matchmaking:" + getAndroidDependencyVersion()};
            }
            public HiveMatchMaking(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class HiveChat : AndroidDependencyXmlInfo {
            public HiveChat() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-chat:" + getAndroidDependencyVersion()};
            }
            public HiveChat(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class HivePromotion : AndroidDependencyXmlInfo {
            public HivePromotion() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-promotion:" + getAndroidDependencyVersion()};
            }
            public HivePromotion(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class HiveRealNameVerification : AndroidDependencyXmlInfo {
            public HiveRealNameVerification() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-real-name-verification:" + getAndroidDependencyVersion()};
            }
            public HiveRealNameVerification(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class MarketAmazon : AndroidDependencyXmlInfo {
            public MarketAmazon() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-iapv4-market-amazon:" + getAndroidDependencyVersion()};
            }
            public MarketAmazon(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                    androidPackageSpec = new List<string>() {$"com.amazon.device:amazon-appstore-sdk:{AndroidExternalLibraryVersion.Market.amazonAppStore}"};
                }
            }
        }

        class HiveAgeRangeAmazon : AndroidDependencyXmlInfo {
            public HiveAgeRangeAmazon() {
                androidRepository = new List<string>() {""};
                androidPackageSpec = new List<string>() {""};
            }
            public HiveAgeRangeAmazon(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class MarketApple : AndroidDependencyXmlInfo {
            public MarketApple() {
                androidRepository = new List<string>() {""};
                androidPackageSpec = new List<string>() {""};
            }
            public MarketApple(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class HiveAgeRangeApple : AndroidDependencyXmlInfo {
            public HiveAgeRangeApple() {
                androidRepository = new List<string>() {""};
                androidPackageSpec = new List<string>() {""};
            }
            public HiveAgeRangeApple(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class MarketGoogle : AndroidDependencyXmlInfo {
            public MarketGoogle() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-iapv4-market-google:" + getAndroidDependencyVersion()};
            }
            public MarketGoogle(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                    androidPackageSpec = new List<string>() {$"com.android.billingclient:billing:{AndroidExternalLibraryVersion.Market.billingClient}"};
                }
            }
        }

        class HiveAgeRangeGoogle : AndroidDependencyXmlInfo {
            public HiveAgeRangeGoogle() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-agerange-google-agesignals:" + getAndroidDependencyVersion()};
            }
            public HiveAgeRangeGoogle(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {$"com.google.android.play:age-signals:{AndroidExternalLibraryVersion.HiveModule.ageSignals}"};
                }
            }
        }

        class MarketHiveStore : AndroidDependencyXmlInfo {
            public MarketHiveStore() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-iapv4-market-hivestore:" + getAndroidDependencyVersion()};
            }
            public MarketHiveStore(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class MarketHuawei : AndroidDependencyXmlInfo {
            public MarketHuawei() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://developer.huawei.com/repo/"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-iapv4-market-huawei:" + getAndroidDependencyVersion()};
            }
            public MarketHuawei(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://developer.huawei.com/repo/"};
                    androidPackageSpec = new List<string>() {$"com.huawei.hms:iap:{AndroidExternalLibraryVersion.Market.hmsIap}"};
                }
            }
        }

        class MarketLebi : AndroidDependencyXmlInfo {
            public MarketLebi() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-iapv4-market-lebi:" + getAndroidDependencyVersion()};
            }
            public MarketLebi(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                    androidPackageSpec = new List<string>() {$"com.alipay.sdk:alipaysdk-android:{AndroidExternalLibraryVersion.Market.alipaySdk}",
                                                            $"com.tencent.mm.opensdk:wechat-sdk-android-without-mta:{AndroidExternalLibraryVersion.Market.wechatSdkMta}"};
                }
            }
        }

        class MarketOneStore : AndroidDependencyXmlInfo {
            public MarketOneStore() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://repo.onestore.co.kr/repository/onestore-sdk-public"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-iapv4-market-onestore:" + getAndroidDependencyVersion()};
            }
            public MarketOneStore(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.onestore.co.kr/repository/onestore-sdk-public"};
                    androidPackageSpec = new List<string>() {$"com.onestorecorp.sdk:sdk-iap:{AndroidExternalLibraryVersion.Market.onestorecorpSdkIap}"};
                }
            }
        }

        class MarketSamsung : AndroidDependencyXmlInfo
        {
            public MarketSamsung()
            {
                androidRepository = new List<string>() { "https://repo.maven.apache.org/maven2" };
                androidPackageSpec = new List<string>() { "com.com2us.android.hive:hive-iapv4-market-samsung:" + getAndroidDependencyVersion() };
            }
            public MarketSamsung(bool sampleBuild) : this()
            {
                if (sampleBuild)
                {
                    androidRepository = new List<string>() { "" };
                    androidPackageSpec = new List<string>() { "" };
                }
            }
        }
        
        class MarketXiaomi : AndroidDependencyXmlInfo
        {
            public MarketXiaomi()
            {
                androidRepository = new List<string>() { "https://repos.xiaomi.com/maven" };
                androidPackageSpec = new List<string>() { "com.com2us.android.hive:hive-iapv4-market-xiaomi-billing:" + getAndroidDependencyVersion() };
            }
            public MarketXiaomi(bool sampleBuild) : this()
            {
                if (sampleBuild)
                {
                    androidRepository = new List<string>() { "https://repos.xiaomi.com/maven" };
                    androidPackageSpec = new List<string>() { $"com.alipay.sdk:alipaysdk-android:{AndroidExternalLibraryVersion.Market.alipaySdk}",
                                                            $"com.google.code.gson:gson:{AndroidExternalLibraryVersion.Auth.gson}",
                                                            $"com.google.zxing:core:{AndroidExternalLibraryVersion.Auth.googleZxing}",
                                                            $"com.xiaomi.gamecenter.sdk:mioauth:{AndroidExternalLibraryVersion.Auth.xiaomiOauth}",
                                                            $"com.xiaomi.gamecenter.sdk:onetrack-sdk:{AndroidExternalLibraryVersion.Auth.xiaomiOnetrack}" };
                }
            }
        }

        class ProviderAdjust : AndroidDependencyXmlInfo {
            public ProviderAdjust() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-analytics-provider-adjust:" + getAndroidDependencyVersion()};
            }
            public ProviderAdjust(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                    androidPackageSpec = new List<string>() {$"com.adjust.sdk:adjust-android:{AndroidExternalLibraryVersion.Analytics.adjust}"};
                }
            }
        }

        class ProviderAirbridge : AndroidDependencyXmlInfo {
            public ProviderAirbridge() {
                androidRepository = new List<string>() { "https://repo.maven.apache.org/maven2", "https://sdk-download.airbridge.io/maven" };
                androidPackageSpec = new List<string>() { "com.com2us.android.hive:hive-analytics-provider-airbridge:" + getAndroidDependencyVersion() };
            }
            public ProviderAirbridge(bool sampleBuild) : this() {
                if (sampleBuild) {
                    androidRepository = new List<string>() { "https://repo.maven.apache.org/maven2", "https://sdk-download.airbridge.io/maven" };
                    androidPackageSpec = new List<string>() { $"io.airbridge:sdk-android:{AndroidExternalLibraryVersion.Analytics.airbridge}" };
                }
            }
        }

        class ProviderApple : AndroidDependencyXmlInfo {
            public ProviderApple() {
                androidRepository = new List<string>() {""};
                androidPackageSpec = new List<string>() {""};
            }
            public ProviderApple(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class ProviderAppleSignIn : AndroidDependencyXmlInfo {
            public ProviderAppleSignIn() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-apple-signin:" + getAndroidDependencyVersion()};
            }
            public ProviderAppleSignIn(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class ProviderAppsFlyer : AndroidDependencyXmlInfo {
            public ProviderAppsFlyer() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-analytics-provider-appsflyer:" + getAndroidDependencyVersion()};
            }
            public ProviderAppsFlyer(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                    androidPackageSpec = new List<string>() {$"com.appsflyer:af-android-sdk:{AndroidExternalLibraryVersion.Analytics.appsflyerSdk}"};
                }
            }
        }

        class ProviderFacebook : AndroidDependencyXmlInfo {
            public ProviderFacebook() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-facebook:" + getAndroidDependencyVersion()};
            }
            public ProviderFacebook(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                    androidPackageSpec = new List<string>() {$"com.facebook.android:facebook-core:{AndroidExternalLibraryVersion.Auth.facebook}",
                                                            $"com.facebook.android:facebook-common:{AndroidExternalLibraryVersion.Auth.facebook}",
                                                            $"com.facebook.android:facebook-login:{AndroidExternalLibraryVersion.Auth.facebook}",
                                                            $"com.facebook.android:facebook-share:{AndroidExternalLibraryVersion.Auth.facebook}"};
                }
            }
        }

        class ProviderFirebase : AndroidDependencyXmlInfo {
            public ProviderFirebase() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-analytics-provider-firebase:" + getAndroidDependencyVersion()};
            }
            public ProviderFirebase(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                    androidPackageSpec = new List<string>() {$"com.google.firebase:firebase-analytics:{AndroidExternalLibraryVersion.Analytics.firebase}"};
                }
            }
        }

        class ProviderGoogle : AndroidDependencyXmlInfo {
            public ProviderGoogle() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-google-signin:" + getAndroidDependencyVersion(),
                                                        "com.com2us.android.hive:hive-authv4-provider-google-credential-signin:" + getAndroidDependencyVersion()};
            }
            public ProviderGoogle(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                    androidPackageSpec = new List<string>() {$"androidx.credentials:credentials:{AndroidExternalLibraryVersion.Auth.credentials}",
                                                            $"androidx.credentials:credentials-play-services-auth:{AndroidExternalLibraryVersion.Auth.credentialPlayServicesAuth}",
                                                            $"com.google.android.libraries.identity.googleid:googleid:{AndroidExternalLibraryVersion.Auth.identityGoogleId}",
                                                            $"com.google.android.gms:play-services-auth:{AndroidExternalLibraryVersion.Auth.playServicesAuth}",
                                                            $"com.google.android.gms:play-services-ads-identifier:{AndroidExternalLibraryVersion.Auth.playServicesAdsIdentifier}",
                                                            $"com.google.android.gms:play-services-analytics:{AndroidExternalLibraryVersion.Auth.playServicesAnalytics}",
                                                            $"com.google.firebase:firebase-messaging:{AndroidExternalLibraryVersion.Auth.firebaseMessaging}"};
                }
            }
        }

        class ProviderGooglePlayGames : AndroidDependencyXmlInfo {
            public ProviderGooglePlayGames() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-google-playgames:" + getAndroidDependencyVersion()};
            }
            public ProviderGooglePlayGames(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com", "https://maven.google.com"};
                    androidPackageSpec = new List<string>() {$"com.google.android.gms:play-services-games-v2:{AndroidExternalLibraryVersion.Auth.playServicesGamesV2}",
                                                            $"com.google.android.gms:play-services-ads-identifier:{AndroidExternalLibraryVersion.Auth.playServicesAdsIdentifier}",
                                                            $"com.google.android.gms:play-services-analytics:{AndroidExternalLibraryVersion.Auth.playServicesAnalytics}",
                                                            $"com.google.firebase:firebase-messaging:{AndroidExternalLibraryVersion.Auth.firebaseMessaging}"};
                }
            }
        }

        class ProviderHiveMembership : AndroidDependencyXmlInfo {
            public ProviderHiveMembership() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-hive-membership:" + getAndroidDependencyVersion()};
            }
            public ProviderHiveMembership(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class ProviderHuawei : AndroidDependencyXmlInfo {
            public ProviderHuawei() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://developer.huawei.com/repo/"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-huawei:" + getAndroidDependencyVersion()};
            }
            public ProviderHuawei(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://developer.huawei.com/repo/"};
                    androidPackageSpec = new List<string>() {$"com.huawei.hms:hwid:{AndroidExternalLibraryVersion.Auth.hmsHwid}"};
                }
            }
        }

        class ProviderLine : AndroidDependencyXmlInfo {
            public ProviderLine() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-line:" + getAndroidDependencyVersion()};
            }
            public ProviderLine(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                    androidPackageSpec = new List<string>() {$"com.linecorp.linesdk:linesdk:{AndroidExternalLibraryVersion.Auth.line}"};
                }
            }
        }

        class ProviderQQ : AndroidDependencyXmlInfo {
            public ProviderQQ() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-qq:" + getAndroidDependencyVersion()};
            }
            public ProviderQQ(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                    androidPackageSpec = new List<string>() {$"com.squareup.okhttp3:okhttp:{AndroidExternalLibraryVersion.Auth.qqOkhttp}"};
                }
            }
        }

        class ProviderRecaptcha : AndroidDependencyXmlInfo {
            public ProviderRecaptcha() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-google-recaptcha:" + getAndroidDependencyVersion()};
            }
            public ProviderRecaptcha(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                    androidPackageSpec = new List<string>() {$"com.google.android.recaptcha:recaptcha:{AndroidExternalLibraryVersion.HiveModule.recaptcha}"};
                }
            }
        }

        class ProviderSingular : AndroidDependencyXmlInfo {
            public ProviderSingular() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com", "https://maven.singular.net"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-analytics-provider-singular:" + getAndroidDependencyVersion()};
            }
            public ProviderSingular(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com", "https://maven.singular.net"};
                    androidPackageSpec = new List<string>() {$"com.singular.sdk:singular_sdk:{AndroidExternalLibraryVersion.Analytics.singular}"};
                }
            }
        }

        class ProviderVK : AndroidDependencyXmlInfo {
            public ProviderVK() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-vk:" + getAndroidDependencyVersion()};
            }
            public ProviderVK(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() { "" };
                    androidPackageSpec = new List<string>() { "" };
                }
            }
        }

        class ProviderWechat : AndroidDependencyXmlInfo {
            public ProviderWechat() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-wechat:" + getAndroidDependencyVersion()};
            }
            public ProviderWechat(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com"};
                    androidPackageSpec = new List<string>() {$"com.tencent.mm.opensdk:wechat-sdk-android-without-mta:{AndroidExternalLibraryVersion.Auth.wechatSdkMta}"};
                }
            }
        }

        class ProviderWeverse : AndroidDependencyXmlInfo {
            public ProviderWeverse() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-weverse:" + getAndroidDependencyVersion()};
            }
            public ProviderWeverse(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class ProviderSteam : AndroidDependencyXmlInfo {
            public ProviderSteam() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-steam:" + getAndroidDependencyVersion()};
            }
            public ProviderSteam(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class ProviderX : AndroidDependencyXmlInfo {
            public ProviderX() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-x:" + getAndroidDependencyVersion()};
            }
            public ProviderX(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class ProviderTelegram : AndroidDependencyXmlInfo
        {
            public ProviderTelegram()
            {
                androidRepository = new List<string>() { "https://repo.maven.apache.org/maven2" };
                androidPackageSpec = new List<string>() { "com.com2us.android.hive:hive-authv4-provider-telegram:" + getAndroidDependencyVersion() };
            }
            public ProviderTelegram(bool sampleBuild) : this()
            {
                if (sampleBuild)
                {
                    androidRepository = new List<string>() { "" };
                    androidPackageSpec = new List<string>() { "" };
                }
            }
        }
        
        class ProviderXiaomi : AndroidDependencyXmlInfo {
            public ProviderXiaomi()
            {
                androidRepository = new List<string>() {"https://repos.xiaomi.com/maven"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-authv4-provider-xiaomi-oauth:" + getAndroidDependencyVersion()};
            }
            public ProviderXiaomi(bool sampleBuild) : this()
            {
                if(sampleBuild)
                {
                    androidRepository = new List<string>() {"https://repos.xiaomi.com/maven"};
                    androidPackageSpec = new List<string>() {$"com.alipay.sdk:alipaysdk-android:{AndroidExternalLibraryVersion.Market.alipaySdk}",
                                                            $"com.google.code.gson:gson:{AndroidExternalLibraryVersion.Auth.gson}",
                                                            $"com.google.zxing:core:{AndroidExternalLibraryVersion.Auth.googleZxing}",
                                                            $"com.xiaomi.gamecenter.sdk:mioauth:{AndroidExternalLibraryVersion.Auth.xiaomiOauth}",
                                                            $"com.xiaomi.gamecenter.sdk:onetrack-sdk:{AndroidExternalLibraryVersion.Auth.xiaomiOnetrack}" };
                }
            }
        }

        class PushAmazonAdm : AndroidDependencyXmlInfo {
            public PushAmazonAdm() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-push-amazon-adm:" + getAndroidDependencyVersion()};
            }
            public PushAmazonAdm(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {""};
                    androidPackageSpec = new List<string>() {""};
                }
            }
        }

        class PushGoogleFcm : AndroidDependencyXmlInfo {
            public PushGoogleFcm() {
                androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2"};
                androidPackageSpec = new List<string>() {"com.com2us.android.hive:hive-push-google-fcm:" + getAndroidDependencyVersion()};
            }
            public PushGoogleFcm(bool sampleBuild) : this() {
                if(sampleBuild) {
                    androidRepository = new List<string>() {"https://repo.maven.apache.org/maven2", "https://jcenter.bintray.com/"};
                    androidPackageSpec = new List<string>() {$"com.google.firebase:firebase-messaging:{AndroidExternalLibraryVersion.Push.firebaseMessaging}"};
                }
            }
        }
    }
}
